<?php
/**
 * ============================================================================
 * classes/Post.php - Post Management Class
 * ============================================================================
 */

namespace App;

class Post {
    private $conn;
    private $table_name = "posts";
    
    public function __construct($db) {
        $this->conn = $db;
    }
    
    /**
     * Create a new post
     */
    public function create($userId, $content, $postType = 'text', $mediaUrls = null, $visibility = 'public', $location = null) {
        $query = "INSERT INTO posts 
                  (user_id, content, post_type, media_urls, visibility, location) 
                  VALUES (:user_id, :content, :post_type, :media_urls, :visibility, :location)";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':user_id', $userId);
        $stmt->bindParam(':content', $content);
        $stmt->bindParam(':post_type', $postType);
        $stmt->bindParam(':media_urls', $mediaUrls);
        $stmt->bindParam(':visibility', $visibility);
        $stmt->bindParam(':location', $location);
        
        if ($stmt->execute()) {
            return $this->conn->lastInsertId();
        }
        
        return false;
    }
    
    /**
     * Get feed posts for a user
     */
    public function getFeed($userId, $limit = 20, $offset = 0) {
        $query = "SELECT 
                    p.*,
                    u.full_name,
                    u.profile_picture,
                    (SELECT COUNT(*) FROM post_likes pl WHERE pl.post_id = p.id) as like_count,
                    (SELECT COUNT(*) FROM post_likes pl WHERE pl.post_id = p.id AND pl.user_id = :user_id) as user_liked,
                    (SELECT COUNT(*) FROM post_comments pc WHERE pc.post_id = p.id) as comment_count
                FROM posts p
                INNER JOIN users u ON p.user_id = u.id
                WHERE (
                    p.visibility = 'public'
                    OR p.user_id = :user_id2
                    OR (p.visibility = 'friends' AND EXISTS (
                        SELECT 1 FROM friendships f 
                        WHERE ((f.user_id = p.user_id AND f.friend_id = :user_id3) 
                        OR (f.friend_id = p.user_id AND f.user_id = :user_id4))
                        AND f.status = 'accepted'
                    ))
                )
                ORDER BY p.is_pinned DESC, p.created_at DESC
                LIMIT :limit OFFSET :offset";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':user_id', $userId);
        $stmt->bindParam(':user_id2', $userId);
        $stmt->bindParam(':user_id3', $userId);
        $stmt->bindParam(':user_id4', $userId);
        $stmt->bindValue(':limit', (int)$limit, \PDO::PARAM_INT);
        $stmt->bindValue(':offset', (int)$offset, \PDO::PARAM_INT);
        $stmt->execute();
        
        return $stmt->fetchAll(\PDO::FETCH_ASSOC);
    }
    
    /**
     * Get user's posts
     */
    public function getUserPosts($userId, $viewerId, $limit = 20, $offset = 0) {
        $query = "SELECT 
                    p.*,
                    u.full_name,
                    u.profile_picture,
                    (SELECT COUNT(*) FROM post_likes pl WHERE pl.post_id = p.id) as like_count,
                    (SELECT COUNT(*) FROM post_likes pl WHERE pl.post_id = p.id AND pl.user_id = :viewer_id) as user_liked,
                    (SELECT COUNT(*) FROM post_comments pc WHERE pc.post_id = p.id) as comment_count
                FROM posts p
                INNER JOIN users u ON p.user_id = u.id
                WHERE p.user_id = :user_id
                AND (
                    p.visibility = 'public'
                    OR p.user_id = :viewer_id2
                    OR (p.visibility = 'friends' AND EXISTS (
                        SELECT 1 FROM friendships f 
                        WHERE ((f.user_id = p.user_id AND f.friend_id = :viewer_id3) 
                        OR (f.friend_id = p.user_id AND f.user_id = :viewer_id4))
                        AND f.status = 'accepted'
                    ))
                )
                ORDER BY p.is_pinned DESC, p.created_at DESC
                LIMIT :limit OFFSET :offset";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':user_id', $userId);
        $stmt->bindParam(':viewer_id', $viewerId);
        $stmt->bindParam(':viewer_id2', $viewerId);
        $stmt->bindParam(':viewer_id3', $viewerId);
        $stmt->bindParam(':viewer_id4', $viewerId);
        $stmt->bindValue(':limit', (int)$limit, \PDO::PARAM_INT);
        $stmt->bindValue(':offset', (int)$offset, \PDO::PARAM_INT);
        $stmt->execute();
        
        return $stmt->fetchAll(\PDO::FETCH_ASSOC);
    }
    
    /**
     * Get single post
     */
    public function getPost($postId, $userId) {
        $query = "SELECT 
                    p.*,
                    u.full_name,
                    u.profile_picture,
                    (SELECT COUNT(*) FROM post_likes pl WHERE pl.post_id = p.id) as like_count,
                    (SELECT COUNT(*) FROM post_likes pl WHERE pl.post_id = p.id AND pl.user_id = :user_id) as user_liked,
                    (SELECT COUNT(*) FROM post_comments pc WHERE pc.post_id = p.id) as comment_count
                FROM posts p
                INNER JOIN users u ON p.user_id = u.id
                WHERE p.id = :post_id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':post_id', $postId);
        $stmt->bindParam(':user_id', $userId);
        $stmt->execute();
        
        return $stmt->fetch(\PDO::FETCH_ASSOC);
    }
    
    /**
     * Like a post
     */
    public function like($postId, $userId) {
        $query = "INSERT INTO post_likes (post_id, user_id) 
                  VALUES (:post_id, :user_id)";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':post_id', $postId);
        $stmt->bindParam(':user_id', $userId);
        
        try {
            return $stmt->execute();
        } catch (\PDOException $e) {
            // Duplicate entry
            return false;
        }
    }
    
    /**
     * Unlike a post
     */
    public function unlike($postId, $userId) {
        $query = "DELETE FROM post_likes 
                  WHERE post_id = :post_id AND user_id = :user_id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':post_id', $postId);
        $stmt->bindParam(':user_id', $userId);
        
        return $stmt->execute();
    }
    
    /**
     * Add comment
     */
    public function addComment($postId, $userId, $content, $parentCommentId = null) {
        $query = "INSERT INTO post_comments 
                  (post_id, user_id, content, parent_comment_id) 
                  VALUES (:post_id, :user_id, :content, :parent_comment_id)";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':post_id', $postId);
        $stmt->bindParam(':user_id', $userId);
        $stmt->bindParam(':content', $content);
        $stmt->bindParam(':parent_comment_id', $parentCommentId);
        
        if ($stmt->execute()) {
            return $this->conn->lastInsertId();
        }
        
        return false;
    }
    
    /**
     * Get post comments
     */
    public function getComments($postId, $limit = 20) {
        $query = "SELECT 
                    pc.*,
                    u.full_name,
                    u.profile_picture
                FROM post_comments pc
                INNER JOIN users u ON pc.user_id = u.id
                WHERE pc.post_id = :post_id
                AND pc.parent_comment_id IS NULL
                ORDER BY pc.created_at ASC
                LIMIT :limit";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':post_id', $postId);
        $stmt->bindValue(':limit', (int)$limit, \PDO::PARAM_INT);
        $stmt->execute();
        
        $comments = $stmt->fetchAll(\PDO::FETCH_ASSOC);
        
        // Get replies for each comment
        foreach ($comments as &$comment) {
            $comment['replies'] = $this->getCommentReplies($comment['id']);
        }
        
        return $comments;
    }
    
    /**
     * Get comment replies
     */
    private function getCommentReplies($commentId) {
        $query = "SELECT 
                    pc.*,
                    u.full_name,
                    u.profile_picture
                FROM post_comments pc
                INNER JOIN users u ON pc.user_id = u.id
                WHERE pc.parent_comment_id = :comment_id
                ORDER BY pc.created_at ASC";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':comment_id', $commentId);
        $stmt->execute();
        
        return $stmt->fetchAll(\PDO::FETCH_ASSOC);
    }
    
    /**
     * Delete post
     */
    public function delete($postId, $userId) {
        $query = "DELETE FROM posts 
                  WHERE id = :post_id AND user_id = :user_id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':post_id', $postId);
        $stmt->bindParam(':user_id', $userId);
        
        return $stmt->execute();
    }
    
    /**
     * Update post
     */
    public function update($postId, $userId, $content, $visibility = null) {
        if ($visibility) {
            $query = "UPDATE posts 
                      SET content = :content, visibility = :visibility, updated_at = NOW()
                      WHERE id = :post_id AND user_id = :user_id";
            
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':visibility', $visibility);
        } else {
            $query = "UPDATE posts 
                      SET content = :content, updated_at = NOW()
                      WHERE id = :post_id AND user_id = :user_id";
            
            $stmt = $this->conn->prepare($query);
        }
        
        $stmt->bindParam(':content', $content);
        $stmt->bindParam(':post_id', $postId);
        $stmt->bindParam(':user_id', $userId);
        
        return $stmt->execute();
    }
    
    /**
     * Pin/Unpin post
     */
    public function togglePin($postId, $userId) {
        $query = "UPDATE posts 
                  SET is_pinned = NOT is_pinned 
                  WHERE id = :post_id AND user_id = :user_id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':post_id', $postId);
        $stmt->bindParam(':user_id', $userId);
        
        return $stmt->execute();
    }
}