<?php
/**
 * ============================================================================
 * classes/Friendship.php - Friendship Management Class
 * ============================================================================
 */

namespace App;

use PDO;
use PDOException;

class Friendship {
    private $db;
    private $table = 'friendships';
    
    public function __construct($db) {
        $this->db = $db;
    }
    
    /**
     * Send friend request
     */
    public function sendRequest($userId, $friendId) {
        try {
            // Check if request already exists
            $sql = "SELECT * FROM {$this->table} 
                    WHERE (user_id = ? AND friend_id = ?) 
                    OR (user_id = ? AND friend_id = ?)";
            $stmt = $this->db->prepare($sql);
            $stmt->execute([$userId, $friendId, $friendId, $userId]);
            
            if ($stmt->fetch()) {
                return ['success' => false, 'message' => 'Friend request already exists'];
            }
            
            $sql = "INSERT INTO {$this->table} (user_id, friend_id) VALUES (?, ?)";
            $stmt = $this->db->prepare($sql);
            
            if ($stmt->execute([$userId, $friendId])) {
                // Create notification
                $sql = "INSERT INTO notifications (user_id, title, message, type, link)
                        VALUES (?, ?, ?, 'info', ?)";
                $stmt = $this->db->prepare($sql);
                $stmt->execute([
                    $friendId,
                    'New Friend Request',
                    'You have a new friend request',
                    '/friends/requests.php'
                ]);
                
                return ['success' => true, 'message' => 'Friend request sent'];
            }
            
            return ['success' => false, 'message' => 'Failed to send request'];
            
        } catch (PDOException $e) {
            error_log("Friend request error: " . $e->getMessage());
            return ['success' => false, 'message' => 'Database error occurred'];
        }
    }
    
    /**
     * Accept friend request
     */
    public function acceptRequest($friendshipId, $userId) {
        try {
            $sql = "UPDATE {$this->table} 
                    SET status = 'accepted', accepted_at = CURRENT_TIMESTAMP
                    WHERE id = ? AND friend_id = ? AND status = 'pending'";
            
            $stmt = $this->db->prepare($sql);
            
            if ($stmt->execute([$friendshipId, $userId])) {
                return ['success' => true, 'message' => 'Friend request accepted'];
            }
            
            return ['success' => false, 'message' => 'Failed to accept request'];
            
        } catch (PDOException $e) {
            return ['success' => false, 'message' => 'Database error occurred'];
        }
    }
    
    /**
     * Reject/delete friend request
     */
    public function deleteRequest($friendshipId, $userId) {
        $sql = "DELETE FROM {$this->table} 
                WHERE id = ? AND (user_id = ? OR friend_id = ?)";
        
        $stmt = $this->db->prepare($sql);
        
        if ($stmt->execute([$friendshipId, $userId, $userId])) {
            return ['success' => true, 'message' => 'Request deleted'];
        }
        
        return ['success' => false, 'message' => 'Failed to delete request'];
    }
    
    /**
     * Get user's friends
     */
    public function getFriends($userId) {
        $sql = "SELECT 
                CASE 
                    WHEN f.user_id = ? THEN f.friend_id
                    ELSE f.user_id
                END as friend_user_id,
                u.full_name, u.profile_picture, u.district, u.user_type,
                f.accepted_at
                FROM {$this->table} f
                JOIN users u ON (
                    CASE 
                        WHEN f.user_id = ? THEN f.friend_id = u.id
                        ELSE f.user_id = u.id
                    END
                )
                WHERE (f.user_id = ? OR f.friend_id = ?)
                AND f.status = 'accepted'
                ORDER BY f.accepted_at DESC";
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute([$userId, $userId, $userId, $userId]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    /**
     * Get pending requests
     */
    public function getPendingRequests($userId) {
        $sql = "SELECT f.*, u.full_name, u.profile_picture, u.district
                FROM {$this->table} f
                JOIN users u ON f.user_id = u.id
                WHERE f.friend_id = ? AND f.status = 'pending'
                ORDER BY f.requested_at DESC";
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute([$userId]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    /**
     * Check friendship status
     */
    public function getStatus($userId, $friendId) {
        $sql = "SELECT status, 
                CASE WHEN user_id = ? THEN 'sent' ELSE 'received' END as direction
                FROM {$this->table}
                WHERE (user_id = ? AND friend_id = ?) 
                OR (user_id = ? AND friend_id = ?)";
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute([$userId, $userId, $friendId, $friendId, $userId]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
    
    /**
     * Search for farmers
     */
    public function searchFarmers($userId, $search = '', $district = '') {
        $sql = "SELECT u.id, u.full_name, u.profile_picture, u.district, u.subcounty,
                (SELECT status FROM {$this->table} 
                 WHERE (user_id = ? AND friend_id = u.id) 
                 OR (user_id = u.id AND friend_id = ?)) as friendship_status
                FROM users u
                WHERE u.id != ?
                AND u.user_type = 'farmer'
                AND u.is_active = 1";
        
        $params = [$userId, $userId, $userId];
        
        if ($search) {
            $sql .= " AND (u.full_name LIKE ? OR u.phone_number LIKE ?)";
            $params[] = "%{$search}%";
            $params[] = "%{$search}%";
        }
        
        if ($district) {
            $sql .= " AND u.district = ?";
            $params[] = $district;
        }
        
        $sql .= " LIMIT 50";
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
}
?>
