<?php
/**
 * ============================================================================
 * groups/index.php - Groups Main Page (Improved Integration)
 * ============================================================================
 */
define('AIMS_ACCESS', true);
require_once __DIR__ . '/../config/config.php';

requireLogin();

$pageTitle = 'Groups';
$userId = getUserId();

$database = new Database();
$db = $database->getConnection();

// Get user's groups from CHATS table (compatible with existing chat system)
$sql = "SELECT c.*, 
        cm.role as user_role,
        (SELECT COUNT(*) FROM chat_members WHERE chat_id = c.id) as member_count,
        (SELECT COUNT(*) FROM messages WHERE chat_id = c.id) as posts_count,
        (SELECT content FROM messages WHERE chat_id = c.id ORDER BY created_at DESC LIMIT 1) as last_message,
        (SELECT created_at FROM messages WHERE chat_id = c.id ORDER BY created_at DESC LIMIT 1) as last_message_time,
        (SELECT COUNT(*) FROM messages m 
         WHERE m.chat_id = c.id 
         AND m.created_at > COALESCE((SELECT last_read_at FROM chat_members WHERE chat_id = c.id AND user_id = ?), '2000-01-01')) as unread_count
        FROM chats c
        INNER JOIN chat_members cm ON c.id = cm.chat_id
        WHERE cm.user_id = ? AND c.chat_type = 'group'
        ORDER BY last_message_time DESC NULLS LAST, c.created_at DESC";
$stmt = $db->prepare($sql);
$stmt->execute([$userId, $userId]);
$myGroups = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Get suggested groups (public groups user hasn't joined)
$sql = "SELECT c.*,
        (SELECT COUNT(*) FROM chat_members WHERE chat_id = c.id) as member_count,
        (SELECT COUNT(*) FROM messages WHERE chat_id = c.id) as posts_count,
        u.full_name as creator_name
        FROM chats c
        LEFT JOIN users u ON c.created_by = u.id
        WHERE c.chat_type = 'group'
        AND c.id NOT IN (
            SELECT chat_id FROM chat_members WHERE user_id = ?
        )
        ORDER BY member_count DESC, c.created_at DESC
        LIMIT 12";
$stmt = $db->prepare($sql);
$stmt->execute([$userId]);
$suggestedGroups = $stmt->fetchAll(PDO::FETCH_ASSOC);

include_once __DIR__ . '/../includes/meta.php';
include_once __DIR__ . '/../includes/header.php';
?>

<style>
.group-card {
    border-radius: 10px;
    overflow: hidden;
    transition: transform 0.2s, box-shadow 0.2s;
    cursor: pointer;
}

.group-card:hover {
    transform: translateY(-5px);
    box-shadow: 0 4px 12px rgba(0,0,0,0.15);
}

.group-cover {
    height: 120px;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-size: 2rem;
    position: relative;
    overflow: hidden;
}

.group-avatar {
    width: 70px;
    height: 70px;
    border: 3px solid white;
    border-radius: 10px;
    margin-top: -35px;
    object-fit: cover;
}

.stats-badge {
    display: inline-flex;
    align-items: center;
    gap: 5px;
    padding: 4px 12px;
    background-color: #f1f3f5;
    border-radius: 20px;
    font-size: 0.85rem;
}

.unread-indicator {
    position: absolute;
    top: 10px;
    right: 10px;
    background: #dc3545;
    color: white;
    border-radius: 12px;
    padding: 4px 10px;
    font-size: 0.75rem;
    font-weight: bold;
}

.filter-sidebar {
    position: sticky;
    top: 20px;
}
</style>

<div class="container py-4">
    <!-- Header -->
    <div class="row mb-4">
        <div class="col-12">
            <div class="d-flex justify-content-between align-items-center flex-wrap gap-3">
                <div>
                    <h2 class="mb-1">
                        <i class="bi bi-people-fill text-primary"></i> Groups
                    </h2>
                    <p class="text-muted mb-0">Connect with farmers who share your interests</p>
                </div>
                <div>
                    <a href="<?php echo APP_URL; ?>/groups/create.php" class="btn btn-primary">
                        <i class="bi bi-plus-circle"></i> Create New Group
                    </a>
                </div>
            </div>
        </div>
    </div>

    <div class="row">
        <!-- Sidebar -->
        <div class="col-md-4 col-lg-3 mb-4">
            <div class="card border-0 shadow-sm filter-sidebar">
                <div class="card-body">
                    <h6 class="mb-3">
                        <i class="bi bi-funnel"></i> Filter Groups
                    </h6>
                    <div class="list-group list-group-flush">
                        <a href="<?php echo APP_URL; ?>/groups/" 
                           class="list-group-item list-group-item-action active">
                            <i class="bi bi-collection me-2"></i>All Groups
                        </a>
                        <a href="<?php echo APP_URL; ?>/groups/?filter=my-groups" 
                           class="list-group-item list-group-item-action">
                            <i class="bi bi-person-check me-2"></i>My Groups
                            <span class="badge bg-primary float-end"><?php echo count($myGroups); ?></span>
                        </a>
                        <a href="<?php echo APP_URL; ?>/groups/?filter=discover" 
                           class="list-group-item list-group-item-action">
                            <i class="bi bi-compass me-2"></i>Discover
                            <span class="badge bg-success float-end"><?php echo count($suggestedGroups); ?></span>
                        </a>
                    </div>
                    
                    <hr>
                    
                    <h6 class="mb-3">
                        <i class="bi bi-lightning-fill"></i> Quick Actions
                    </h6>
                    <div class="d-grid gap-2">
                        <a href="<?php echo APP_URL; ?>/chat/" class="btn btn-sm btn-outline-primary">
                            <i class="bi bi-chat-dots"></i> All Messages
                        </a>
                        <a href="<?php echo APP_URL; ?>/friends/find-farmers.php" class="btn btn-sm btn-outline-success">
                            <i class="bi bi-person-plus"></i> Find Farmers
                        </a>
                    </div>
                </div>
            </div>
        </div>

        <!-- Main Content -->
        <div class="col-md-8 col-lg-9">
            <!-- My Groups Section -->
            <?php if (!empty($myGroups)): ?>
                <div class="mb-4">
                    <div class="d-flex justify-content-between align-items-center mb-3">
                        <h5 class="mb-0">
                            <i class="bi bi-collection-fill text-primary"></i> My Groups
                        </h5>
                        <a href="<?php echo APP_URL; ?>/chat/" class="btn btn-sm btn-link">
                            View in Messages <i class="bi bi-arrow-right"></i>
                        </a>
                    </div>
                    
                    <div class="row g-3 mb-4">
                        <?php foreach ($myGroups as $group): ?>
                            <div class="col-md-6 col-lg-4">
                                <div class="card border-0 shadow-sm group-card h-100"
                                     onclick="window.location.href='<?php echo APP_URL; ?>/groups/view.php?id=<?php echo $group['id']; ?>'">
                                    <div class="group-cover">
                                        <?php if ($group['unread_count'] > 0): ?>
                                            <span class="unread-indicator">
                                                <?php echo $group['unread_count']; ?> new
                                            </span>
                                        <?php endif; ?>
                                        <i class="bi bi-people-fill" style="opacity: 0.3;"></i>
                                    </div>
                                    <div class="card-body text-center">
                                        <?php if ($group['chat_avatar']): ?>
                                            <img src="<?php echo htmlspecialchars($group['chat_avatar']); ?>" 
                                                 class="group-avatar mb-3"
                                                 alt="<?php echo htmlspecialchars($group['chat_name']); ?>">
                                        <?php else: ?>
                                            <div class="group-avatar bg-light d-flex align-items-center justify-content-center mb-3 mx-auto">
                                                <i class="bi bi-people-fill text-muted" style="font-size: 1.8rem;"></i>
                                            </div>
                                        <?php endif; ?>
                                        
                                        <h6 class="mb-2">
                                            <?php echo htmlspecialchars($group['chat_name']); ?>
                                        </h6>
                                        
                                        <?php if ($group['last_message']): ?>
                                            <p class="small text-muted mb-2 text-truncate">
                                                <?php echo truncate($group['last_message'], 50); ?>
                                            </p>
                                            <small class="text-muted">
                                                <i class="bi bi-clock"></i>
                                                <?php echo timeAgo($group['last_message_time']); ?>
                                            </small>
                                        <?php else: ?>
                                            <p class="small text-muted fst-italic mb-2">No messages yet</p>
                                        <?php endif; ?>
                                        
                                        <hr class="my-2">
                                        
                                        <div class="d-flex justify-content-center gap-2 mb-3">
                                            <span class="stats-badge">
                                                <i class="bi bi-people"></i>
                                                <?php echo $group['member_count']; ?>
                                            </span>
                                            <span class="stats-badge">
                                                <i class="bi bi-chat-dots"></i>
                                                <?php echo $group['posts_count']; ?>
                                            </span>
                                        </div>
                                        
                                        <?php if ($group['user_role'] === 'admin'): ?>
                                            <span class="badge bg-success">
                                                <i class="bi bi-shield-check"></i> Admin
                                            </span>
                                        <?php endif; ?>
                                    </div>
                                    <div class="card-footer bg-transparent border-0 pt-0">
                                        <div class="d-grid gap-2">
                                            <a href="<?php echo APP_URL; ?>/groups/view.php?id=<?php echo $group['id']; ?>" 
                                               class="btn btn-sm btn-outline-primary"
                                               onclick="event.stopPropagation();">
                                                <i class="bi bi-chat-dots"></i> Open Chat
                                            </a>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>
            <?php else: ?>
                <div class="alert alert-info mb-4">
                    <div class="d-flex align-items-center">
                        <i class="bi bi-info-circle fs-3 me-3"></i>
                        <div>
                            <h6 class="alert-heading mb-1">No Groups Yet</h6>
                            <p class="mb-0">You haven't joined any groups. Discover groups below or create your own!</p>
                        </div>
                    </div>
                </div>
            <?php endif; ?>

            <!-- Suggested Groups Section -->
            <?php if (!empty($suggestedGroups)): ?>
                <div>
                    <div class="d-flex justify-content-between align-items-center mb-3">
                        <h5 class="mb-0">
                            <i class="bi bi-compass-fill text-success"></i> Discover Groups
                        </h5>
                    </div>
                    
                    <div class="row g-3">
                        <?php foreach ($suggestedGroups as $group): ?>
                            <div class="col-md-6 col-lg-4">
                                <div class="card border-0 shadow-sm group-card h-100">
                                    <div class="group-cover" style="background: linear-gradient(135deg, <?php echo sprintf('#%06X', mt_rand(0, 0xFFFFFF)); ?> 0%, <?php echo sprintf('#%06X', mt_rand(0, 0xFFFFFF)); ?> 100%);">
                                        <i class="bi bi-people-fill" style="opacity: 0.3;"></i>
                                    </div>
                                    <div class="card-body text-center">
                                        <?php if ($group['chat_avatar']): ?>
                                            <img src="<?php echo htmlspecialchars($group['chat_avatar']); ?>" 
                                                 class="group-avatar mb-3"
                                                 alt="<?php echo htmlspecialchars($group['chat_name']); ?>">
                                        <?php else: ?>
                                            <div class="group-avatar bg-light d-flex align-items-center justify-content-center mb-3 mx-auto">
                                                <i class="bi bi-people-fill text-muted" style="font-size: 1.8rem;"></i>
                                            </div>
                                        <?php endif; ?>
                                        
                                        <h6 class="mb-2">
                                            <?php echo htmlspecialchars($group['chat_name']); ?>
                                        </h6>
                                        
                                        <?php if ($group['creator_name']): ?>
                                            <p class="small text-muted mb-2">
                                                <i class="bi bi-person-circle"></i>
                                                by <?php echo htmlspecialchars($group['creator_name']); ?>
                                            </p>
                                        <?php endif; ?>
                                        
                                        <hr class="my-2">
                                        
                                        <div class="d-flex justify-content-center gap-2 mb-3">
                                            <span class="stats-badge">
                                                <i class="bi bi-people"></i>
                                                <?php echo $group['member_count']; ?>
                                            </span>
                                            <span class="stats-badge">
                                                <i class="bi bi-chat-dots"></i>
                                                <?php echo $group['posts_count']; ?>
                                            </span>
                                        </div>
                                        
                                        <small class="text-muted d-block mb-3">
                                            <i class="bi bi-calendar"></i>
                                            Created <?php echo timeAgo($group['created_at']); ?>
                                        </small>
                                    </div>
                                    <div class="card-footer bg-transparent border-0 pt-0">
                                        <div class="d-grid gap-2">
                                            <form method="POST" action="<?php echo APP_URL; ?>/groups/join.php">
                                                <input type="hidden" name="group_id" value="<?php echo $group['id']; ?>">
                                                <button type="submit" class="btn btn-sm btn-success w-100">
                                                    <i class="bi bi-box-arrow-in-right"></i> Join Group
                                                </button>
                                            </form>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>
            <?php else: ?>
                <div class="text-center py-5">
                    <i class="bi bi-search" style="font-size: 4rem; color: #dee2e6;"></i>
                    <h4 class="mt-3 text-muted">No Groups to Discover</h4>
                    <p class="text-muted">Be the first to create a farming group!</p>
                    <a href="<?php echo APP_URL; ?>/groups/create.php" class="btn btn-primary mt-2">
                        <i class="bi bi-plus-circle"></i> Create Group
                    </a>
                </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<?php
include_once __DIR__ . '/../includes/footer.php';
include_once __DIR__ . '/../includes/bottom-nav.php';
?>